//
//  GeometryGamesView.swift
//
//  Created by Jeff on 5/10/20.
//  Copyright © 2020 Jeff Weeks. All rights reserved.
//

import UIKit
import Metal
import CoreMotion


class GeometryGamesView: UIView, UIDocumentPickerDelegate {

	let itsModelData: GeometryGamesModel

	let itsDevice: MTLDevice
	let itsRenderer: GeometryGamesRenderer
	let itsExportedImagesHaveTransparentBackground: Bool

	//	itsAnimationTimer keeps a strong reference to its target,
	//	namely this GeometryGamesView.  If this GeometryGamesViewView
	//	also kept a strong reference to itsAnimationTimer,
	//	we'd have a strong reference cycle and neither object
	//	would ever get deallocated.  So keep a weak reference instead.
	//	The run loop will keep a strong reference to itsAnimationTimer
	//	as long as its valid, so there's no danger
	//	of it getting deallocated too soon.
	//
	//		Note:  Because itsAnimationTimer retains this GeometryGamesViewView,
	//		the GeometryGamesViewView would never get deallocated
	//		if we merely paused itsAnimationTimer when not needed.
	//		Instead we must invalidate (and therefore deallocate)
	//		the timer when not needed, and recreate it when needed again.
	//
	weak var itsAnimationTimer: CADisplayLink? = nil
		
	//	When the view size changes, we manually adjust
	//	the drawable size accordingly, and then need
	//	to request a redraw.
	var itsDrawableSizeHasChanged: Bool = false
	
	//	What was the value of the ModelData's change count
	//	the last time we refreshed our view(s)?
	//
	//		Note:  The ModelData initializes itsChangeCount to 0,
	//		so we should initialize itsPreviousChangeCount
	//		to a different value to ensure an initial update.
	//
	var itsPreviousChangeCount: UInt64 = 0xFFFFFFFFFFFFFFFF


	override class var layerClass: AnyClass {
		return CAMetalLayer.self}


	init?(
		frame frameRect: CGRect,
		modelData: GeometryGamesModel,
		wantsMultisampling: Bool,
		wantsDepthBuffer: Bool,
		clearColor: MTLClearColor,
		exportsWithTransparentBackground: Bool,
		rendererSubclass: GeometryGamesRenderer.Type
	) {
	
		itsModelData = modelData
		
		guard let theDevice = MTLCreateSystemDefaultDevice() else {
			return nil
		}
		itsDevice = theDevice

		//	Extended range pixel formats
		//
		//		Apple's "extended range" pixel formats
		//
		//			bgr10_xr
		//			bgr10_xr_srgb
		//
		//		use sRGB color-space coordinates, but allow the color components
		//		to extend beyond the usual [0.0, 1.0] range, giving us access
		//		to wider color spaces like Display P3.
		//
		//		Any iDevice in the MTLGPUFamilyApple3 GPU family
		//		or higher supports extended range pixel formats
		//		when paired with a wide-color display.
		//		All such devices support iOS 13, while no earlier devices do,
		//		so by requiring iOS 13 or later (which SwiftUI needs
		//		in any case) we ensure that extended range pixel formats
		//		are available when needed.  When running on a device
		//		with a non-wide-color display, we may use bgra8Unorm_srgb instead.
		//
		//		Conclusion:  Extended range pixel formats are a big win,
		//		and all newly revised Geometry Games apps use them
		//		whenever the display supports wide color.
		//
		//	sRGB gamma decoding
		//
		//		The two pixel formats
		//
		//			bgr10_xr
		//			bgr10_xr_srgb
		//
		//		can both be used with the full Display P3 gamut.
		//		Somewhat counter-intuitively, the "_srgb" suffix
		//		doesn't mean that we're restricting to the sRGB gamut.
		//		Instead, it means that we're asking Metal to take
		//		the (possibly extended) linear sRGB values that we give it
		//		and automatically gamma-encode them before writing them
		//		into the frame buffer and, conversely, automatically
		//		gamma-decoding those values when reading them
		//		from the frame buffer.  Both Display P3 and sRGB
		//		use the same encoding function -- known as the
		//		"sRGB gamma-encoding function" -- which is why
		//		the suffix "_sRGB" is used to denote a request
		//		for this automatic gamma-encoding and -decoding.
		//
		//		When using a pixel format without the "_sRGB" suffix,
		//		Metal assumes we'll take responsibility for doing
		//		the encoding ourselves, for example in our fragment function.
		//		If we fail to do so, the mid-tones in the image will
		//		look too dark.  For example, if we pass a color component
		//		of 0.5, intending it as a linear value, and Metal
		//		interprets it as a gamma-encoded value of 0.5,
		//		it will come out as dark as a linear value of ~0.22.
		//
		//		Given that gamma-encoding is needed no matter what,
		//		we might as well let Metal take care of it automatically.
		//		So all newly revised Geometry Games apps use an _srgb pixel format.
		//
		//			Boring technical comment:  When I try reading a texture
		//			from the Asset Catalog as a Texture Set, the colors
		//			come out too dark when the frame buffer has
		//			a plain (non _srgb) pixel format, but come out correct
		//			when the frame buffer has an _srgb pixel format.
		//			So that's another reason to prefer an _srgb pixel format.
		//
		//			Unfounded speculation:  I'm guessing that gamma-encoding
		//			is inexpensive (perhaps with hardware support) and of course
		//			the GPU can keep linearized framebuffer data in tile memory,
		//			and gamma-encoding it just before writing it to system memory
		//			(but I don't really know whether it does that or not).
		//			In any case, gamma-encoded data is ultimately what
		//			the display will want to show.
		//
		//	Transparency
		//
		//		Even in apps that make extensive use of partial transparency
		//		while rendering, it's fine for the underlying frame buffer
		//		to be opaque.  The only exception would be if we want to export
		//		an image with a transparent background, in which case we may
		//		use the 64-bit pixel format
		//
		//			bgra10_xr_srgb
		//
		//		or alternatively
		//
		//			rgba16Float
		//
		//		also gives access to extended range sRGB color coordinates.
		//		Somewhat surprisingly, rgba16Float handles the gamma-encoding
		//		correctly, even though it has no _srgb suffix.
		//		I'm guessing the reason that Apple recommends bgra10_xr_srgb
		//		instead of rgba16Float is that, according to Apple,
		//		the display can use the former's 10-bit encoded
		//		color components directly.
		//
#if targetEnvironment(simulator)
		//	Apple Silicon Macs let free-standing iOS apps
		//	use extended-range pixel format, but the Simulator does not.
		//	The page
		//		https://stackoverflow.com/questions/47721708/display-p3-screenshots-from-ios-simulator
		//	says
		//
		//		Unfortunately, the QuartzCore software renderer only supports sRGB.
		//		There is no way to get extended range sRGB or P3 out of that render pipeline
		//		in the simulator.
		//
		let theColorPixelFormat = MTLPixelFormat.rgba16Float
#else
		let theColorPixelFormat = (
			UIScreen.main.traitCollection.displayGamut == UIDisplayGamut.P3 ?	//	wide color?
			(
				exportsWithTransparentBackground ?
					MTLPixelFormat.bgra10_xr_srgb :	//	64-bit format
					MTLPixelFormat.bgr10_xr_srgb	//	32-bit format
			) :
			MTLPixelFormat.bgra8Unorm_srgb
		)
#endif
		let theDepthPixelFormat = (wantsDepthBuffer ?
									MTLPixelFormat.depth32Float :
									MTLPixelFormat.invalid)
		let theSampleCount = UInt(wantsMultisampling ? 4 : 1)

		guard let theRenderer = rendererSubclass.init(
									modelData: itsModelData,
									device: itsDevice,
									colorPixelFormat: theColorPixelFormat,
									depthPixelFormat: theDepthPixelFormat,
									sampleCount: theSampleCount,
									clearColor: clearColor
		) else {
			return nil
		}
		itsRenderer = theRenderer
		
		itsExportedImagesHaveTransparentBackground = exportsWithTransparentBackground

		//	Call super.init().
		//	Among other things, super.init() will create the CAMetalLayer.
		super.init(frame: frameRect)


		//	super.init() has created the CAMetalLayer,
		//	so we may now set the CAMetalLayer's pixel format.
		//
		//		Note:  The pixel format here must agree
		//		with the pixel format passed to itsRenderer.
		//
		if let theCAMetalLayer = self.layer as? CAMetalLayer {
			theCAMetalLayer.device = theDevice
			theCAMetalLayer.pixelFormat = theColorPixelFormat
		}

		//	We'll draw all Metal graphics at the display's native scale.
		//	Please see the comments in layoutSubviews() for further details.
		//	See also Apple's Technical Q&A QA1909
		//
		//			https://developer.apple.com/library/content/qa/qa1909/_index.html
		//
		self.contentScaleFactor = UIScreen.main.nativeScale

		//	Create itsAnimationTimer.
		//
		//		Note:  iOS will automatically stop the timer
		//		when our app is in the background.
		//
		let theAnimationTimer = CADisplayLink(
								target: self,
								selector: #selector(animationTimerFired))
		theAnimationTimer.add(to: RunLoop.current, forMode: RunLoop.Mode.default)
		itsAnimationTimer = theAnimationTimer
	}

	required init(coder: NSCoder) {
		fatalError("init(coder:) has not been implemented")
	}
	
	//	layoutSubviews() is really a "viewSizeDidChange()" message.
	override func layoutSubviews() {
	
		//	The view size has changed, so we must manually
		//	adjust the CAMetalLayer's drawableSize accordingly.

		//	The tricky part is that iOS might be drawing
		//	the user interface elements at some integer scale
		//	(either 2x or 3x) and then resampling the result
		//	at some non-integer scale to fit the device's display.
		//	For example, the iPhone 6+, 6s+, 7+ and 8+ all normally
		//	draw at 3x and then downsample to ~2.6x (1080/414).
		//	Some of the other iPhones, such as my iPhone SE 2020,
		//	normally render at 2x, but also offer a Display Zoom mode
		//	in which a 2x image gets upsampled to ~2.3x (750/320).
		//
		//	Happily, iOS let's us bypass that resampling,
		//	and instead render our Metal image at the device's
		//	native pixel resolution.  To do so, we need to set
		//	theDrawableSize to exactly the size of the area that's
		//	been assigned to our view (measured in native pixels).
		//
		//	Unhappily, iOS doesn't tell us what that required
		//	size (measured in native pixels) is.  Instead, we must
		//	figure it out for ourselves, by computing
		//
		//		theDrawableSize (in pixels)
		//		   = the view size (in points)
		//		   * the content scale factor (in pixels/point)
		//
		//	So far so good.
		//	The only catch is that the computed value of theDrawableSize
		//	typically comes out to a non-integer, and if we round it
		//	in the wrong direction to get an integer, our value
		//	for theDrawableSize might be 1 pixel wider or narrower
		//	(and taller or shorter) than the area iOS has assigned
		//	to our view, in which case iOS might end up re-sampling
		//	our image after all -- which is something we were hoping
		//	to avoid.
		//
		//	In practice, we can pass our computed non-integer value
		//	of theDrawableSize and let iOS do the rounding
		//	(which it does immediately, apparently in the "setter"
		//	for theCAMetalLayer.drawableSize).  Moreover, testing shows
		//	that iOS always rounds down, so for example if we request
		//	a width of 636.87, we'll get 636, not 637.
		//
		//	In most cases, passing the un-rounded value should
		//	let us match the size of the assigned area (in native pixels)
		//	exactly.  The only remaining risk is that, when the computed area
		//	happens to have integer dimensions, the inherent error
		//	in the floating-point value of contentScaleFactor might
		//	cause us to compute, say, a width of 636.999999999 instead
		//	of the required 637, which might force iOS to re-sample our image
		//	because 636.99999999 rounds down to 636, which is 1 less than
		//	the required 637.  To avoid that, we can ignore the floating-point
		//	value contentScaleFactor and instead take the content scale factor
		//	to be the ratio of two integers, namely the main screen's
		//
		//			width in native pixels
		//			----------------------
		//			   width in points
		//
		//	to ensure that integer results always come out exact.

		//	2020-12-11  My iPhone SE 2020 normally renders a 375×667 pt layout
		//	on its native 750×1334 px display, for an exact 2x scale factor.
		//	But it also offers a Display Zoom option, which treats
		//	that same native 750×1334 px display as 320×568 pt layout.
		//	The problem is that the scale factor is anisotropic:
		//
		//		the horizontal scale factor is  750/320 = 2.34375
		//		the  vertical  scale factor is 1334/568 = 2.3485915…
		//
		//	So some extra fussing around is required.

		//	The screen bounds (in points) are given
		//	relative to the current interface orientation.
		let theScreenSizeInPoints = UIScreen.main.bounds.size

		//	By contrast, the nativeBounds (in pixels) are always given
		//	in portrait orientation when running on an iOS device,
		//	but are given in the natural landscape orientation
		//	when running on a Mac.  So in effect at runtime
		//	we don't know whether the nativeBounds are using
		//	the same orientation as the regular "bounds",
		//	or the opposite.  So for simplicity let's just
		//	match long with long and short with short.
		let theScreenSizeInPixels: CGSize
		if ((UIScreen.main.nativeBounds.size.height >= UIScreen.main.nativeBounds.size.width)
		 == (theScreenSizeInPoints.height >= theScreenSizeInPoints.width)
		) {
			theScreenSizeInPixels = UIScreen.main.nativeBounds.size
		} else {
			//	Swap width and height.
			theScreenSizeInPixels = CGSize(
				width:  UIScreen.main.nativeBounds.size.height,
				height: UIScreen.main.nativeBounds.size.width)
		}
		
		//	By doing the multiplication before the division,
		//	we ensure that whenever the result should be an integer,
		//	it will get computed exactly.
		let theDrawableSize = CGSize(
			width:  (self.bounds.size.width  * theScreenSizeInPixels.width ) / theScreenSizeInPoints.width,
			height: (self.bounds.size.height * theScreenSizeInPixels.height) / theScreenSizeInPoints.height)

		if let theCAMetalLayer = self.layer as? CAMetalLayer {
			theCAMetalLayer.drawableSize = theDrawableSize
		}

		//	Request a redraw.
		itsDrawableSizeHasChanged = true
	}


// MARK: -
// MARK: Animation

	@objc func animationTimerFired(displayLink: CADisplayLink) {

		//	Note:  iOS automatically stops the animation timer
		//	when the app is in the background, so we needn't worry
		//	about wasting time drawing a view that isn't visible.
		
		//	Update the model, noting the possibly new change count.
		let theChangeCount = updateModel()
		
		//	Has the model changed?
		let theModelHasChanged = (theChangeCount != itsPreviousChangeCount)
		itsPreviousChangeCount = theChangeCount
		
		//	Has the drawable size changed?
		let theDrawableSizeHasChanged = itsDrawableSizeHasChanged
		itsDrawableSizeHasChanged = false
		
		//	Redraw the view iff something has changed.
		if (theModelHasChanged || theDrawableSizeHasChanged) {
			redrawView()
		}

		//	Play any pending sound.
		PlayPendingSound()
	}
		
	func updateModel() -> UInt64 {	//	returns the model's change count

		let theChangeCount: UInt64

		do {
			let md = itsModelData.lockModelData()
		
			//	Update the ModelData as needed.
			if (SimulationWantsUpdates(md)) {
				let theFrameTime = 1.0 / Double(UIScreen.main.maximumFramesPerSecond)
				SimulationUpdate(md, theFrameTime)	//	increments itsChangeCount
			}

			//	Note the current change count.
			theChangeCount = GetChangeCount(md)
			
			itsModelData.unlockModelData()
		}
		
		return theChangeCount
	}

	func redrawView() {

		//	Wrap the render() call in an autorelease pool
		//	so that the drawable gets released as soon as
		//	the render() function returns.
		//
		autoreleasepool {

			if let theCAMetalLayer = self.layer as? CAMetalLayer {
			
				//	Ask the CAMetalLayer for the next available color buffer.
				//	nextDrawable() blocks until a drawable becomes available.
				//	It returns nil only if
				//		- theCAMetalLayer has an invalid combination of drawable properties,
				//		- all drawables are in use and the 1-second timeout has elapsed, or
				//		- the process is out of memory.
				//
				if let theDrawable = theCAMetalLayer.nextDrawable() {
					itsRenderer.render(drawable: theDrawable)
				}
			}
		}
	}
	

// MARK: -
// MARK: Export

	func copyImage(
		requestedWidthPx: UInt,
		requestedHeightPx: UInt
	) -> (UInt, UInt) {	//	returns rendered size

		let (theOptionalPngRepresentation, theRenderedWidthPx, theRenderedHeightPx)
			= offscreenImageAsPNG(requestedWidthPx: requestedWidthPx, requestedHeightPx: requestedHeightPx)

		guard let thePngRepresentation = theOptionalPngRepresentation else {
			return (0, 0)
		}

		UIPasteboard.general.setData(thePngRepresentation, forPasteboardType: "public.png")

		return (theRenderedWidthPx, theRenderedHeightPx)
	}
		
	func saveImage(
		requestedWidthPx: UInt,
		requestedHeightPx: UInt
	) -> (UInt, UInt) {	//	returns rendered size

		let (theOptionalPngRepresentation, theRenderedWidthPx, theRenderedHeightPx)
			= offscreenImageAsPNG(requestedWidthPx: requestedWidthPx, requestedHeightPx: requestedHeightPx)

		guard let thePngRepresentation = theOptionalPngRepresentation else {
			return (0, 0)
		}

		let (_, theTemporaryImageURL) = temporaryFileURL()

		do {
			try thePngRepresentation.write(to: theTemporaryImageURL)
	
			//	In principle we'd be happy to let
			//	iOS move the temporary file,
			//	with no need to make a copy of it.
			//	The downside of that approach, though,
			//	is that the Document Picker would invite
			//	the user to "Move" the exported image.
			//	We'd much prefer that the Document Picker
			//	invite the user to "Save" the exported image,
			//	and for that to happen we need to ask it
			//	to save a copy of the temporary file.
			let theController = UIDocumentPickerViewController(
									forExporting: [theTemporaryImageURL],
									asCopy: true)
			
			//	In principle we could ignore the temporary file,
			//	and iOS would eventually delete it (because it's
			//	in the temporaryDirectory).  But a cleaner solution
			//	is to make ourselves theController's delegate
			//	and let the delegate callback methods
			//	delete the temporary file immediately.
			theController.delegate = self

			UIApplication.shared.windows.first?.rootViewController?.present(theController, animated: false)
			
		} catch {
			print("Error saving image file")
		}

		return (theRenderedWidthPx, theRenderedHeightPx)
	}

	func documentPicker(_ controller: UIDocumentPickerViewController, didPickDocumentsAt urls: [URL]) {
		deleteTemporaryFile()
	}

	func documentPickerWasCancelled(_ controller: UIDocumentPickerViewController) {
		deleteTemporaryFile()
	}

	func deleteTemporaryFile() {
	
		let (theFileManager, theTemporaryImageURL) = temporaryFileURL()
		do {
			try theFileManager.removeItem(at: theTemporaryImageURL)
		} catch {
			print("Error deleting temporary image file")
		}
	}

	func temporaryFileURL() -> (FileManager, URL) {

		let theFileManager = FileManager.default
		let theTemporaryFileName = Bundle.main.localizedString(
										forKey: "Untitled",
										value: nil,
										table: "GeometryGamesLocalizable"
									).appending(".png")
		let theTemporaryFileURL = theFileManager
									.temporaryDirectory
									.appendingPathComponent(theTemporaryFileName)

		return (theFileManager, theTemporaryFileURL)
	}
	

	func offscreenImageAsPNG(
		requestedWidthPx: UInt,
		requestedHeightPx: UInt
	) -> (Data?, UInt, UInt) {	//	returns (PNG data, rendered width, rendered height)

		let theValidatedWidthPx  = clampImageSize(requestedSizePx: requestedWidthPx )
		let theValidatedHeightPx = clampImageSize(requestedSizePx: requestedHeightPx)

		guard let theCGImage = itsRenderer.createOffscreenImage(
						widthPx:  theValidatedWidthPx,
						heightPx: theValidatedHeightPx,
						withTransparentBackground: itsExportedImagesHaveTransparentBackground
		) else {
			return (nil, 0, 0)
		}

		let theUIImage = UIImage(cgImage: theCGImage)

		guard let thePngRepresentation = theUIImage.pngData() else {
			return (nil, 0, 0)
		}

		return (thePngRepresentation, theValidatedWidthPx, theValidatedHeightPx)
	}

	func clampImageSize(requestedSizePx: UInt) -> UInt {

		let theMaxFramebufferSize = GetMaxFramebufferSizeOnDevice(itsDevice)

		if requestedSizePx == 0 {
			return 1
		}

		if requestedSizePx > theMaxFramebufferSize {
			return theMaxFramebufferSize
		}

		return requestedSizePx
	}


// MARK: -
// MARK: Gesture handling

	func gestureLocationAsDisplayPoint(_ cgPoint: CGPoint) -> DisplayPoint {

		//	Both the cgPoint and theViewBounds should be in points (not pixels),
		//	but either would be fine just so everything is consistent.

		let theViewBounds = self.bounds
		
		var theCGPoint = cgPoint

		//	theCGPoint in coordinates (0,0) to (width, height)
		theCGPoint.x -= theViewBounds.origin.x
		theCGPoint.y -= theViewBounds.origin.y

		//	Flip from iOS's top-down coordinates
		//	to our DisplayPoint's bottom-up coordinates.
		theCGPoint.y = theViewBounds.size.height - theCGPoint.y
		
		//	Package up the result.
		let theResult = DisplayPoint(
			itsX: Double(theCGPoint.x),
			itsY: Double(theCGPoint.y),
			itsViewWidth:  Double(theViewBounds.size.width),
			itsViewHeight: Double(theViewBounds.size.height))

		return theResult
	}

	func gestureTranslationAsDisplayPointMotion(_ translation: CGPoint) -> DisplayPointMotion {

		//	Both the translation and theViewBounds should be in points (not pixels),
		//	but either would be fine just so everything is consistent.

		let theViewBounds = self.bounds
		
		var theTranslation = translation

		//	Flip from iOS's top-down coordinates
		//	to our DisplayPointMotion's bottom-up coordinates.
		theTranslation.y = -theTranslation.y
		
		//	Package up the result.
		let theResult = DisplayPointMotion(
			itsDeltaX: Double(theTranslation.x),
			itsDeltaY: Double(theTranslation.y),
			itsViewWidth:  Double(theViewBounds.size.width),
			itsViewHeight: Double(theViewBounds.size.height))

		return theResult
	}
}
